package com.td4.example.demo1.service;

import com.td4.example.demo1.control.exceptions.HeroNotFoundException;
import com.td4.example.demo1.control.exceptions.ResourceNotFoundException;
import com.td4.example.demo1.model.Hero;
import com.td4.example.demo1.model.DTO.HeroDTO;
import com.td4.example.demo1.model.Power;
import com.td4.example.demo1.repository.HeroesRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Service
public class HeroesService {
    private HeroesRepository heroesRepository;

    @Autowired
    public HeroesService(HeroesRepository heroesRepository) {
        this.heroesRepository = heroesRepository;
    }

    /* *********************************************
       services to get data
    *********************************************** */
    public List<Hero> findAllHeroes() {
        return heroesRepository.findAll();
    }

    public Hero findHeroById(String id) throws ResourceNotFoundException {
        System.out.println("Getting hero by id: " + id);
        Optional<Hero> hero = heroesRepository.findById(id);
        // In case of the hero cannot be found, a HeroNotFoundException is raised. Since it
        // is a subclass of ResourceNotFoundException, there is no need to modify the header
        // of the current method, by adding throws HeroNotFoundException
        if (hero.isEmpty()) throw new HeroNotFoundException("searching for hero with id = "+id);
        return hero.get();
    }

    public Hero findHeroByPublicName(String publicName) throws ResourceNotFoundException {
        System.out.println("Getting hero by public name: " + publicName);
        Optional<Hero> hero = heroesRepository.findByPublicName(publicName);
        // Normally, this method would throw a HeroNotFoundException. But in order to show how errors are
        // managed by the ResourceNotFoundAdvice class, a more general ResourceNotFoundException is thrown
        if (hero.isEmpty()) throw new ResourceNotFoundException("searching for hero with publicName = "+publicName);
        return hero.get();
    }

    public Page<Hero> findHeroesByPublicNameLike(String pattern, Pageable pageable) {
        return heroesRepository.findAllByPublicNameLike(pattern, pageable);
    }


    public List<Hero> findAllByPower(String string, boolean pattern) {
        List<Hero> heroes = null;
        if (pattern) {
            System.out.println("Getting heroes with power containing " + string);
            heroes = heroesRepository.findAllByPowerContaining(string);
        }
        else {
            System.out.println("Getting heroes with power = " + string);
            heroes = heroesRepository.findAllByPower(string);
        }
        return heroes;
    }

    public long count() {
        long count = heroesRepository.count();
        return count;
    }

    /* *********************************************
       services to create data
    *********************************************** */
    public Hero createHero(String publicName, String realName, List<Power> powers) {
        Hero hero = new Hero(publicName, realName, powers);
        heroesRepository.save(hero);
        return hero;
    }

    /* *********************************************
       services to delete data
    *********************************************** */
    public void deleteHero(String id) {
        Optional<Hero> heroOpt = heroesRepository.findById(id);
        if (heroOpt.isEmpty()) throw new HeroNotFoundException("deleting hero with id = "+id);
        heroesRepository.deleteById(id);
    }
}
