package com.td3.example.demoAll.service;

import com.td3.example.demoAll.control.exceptions.HeroNotFoundException;
import com.td3.example.demoAll.control.exceptions.ResourceException;
import com.td3.example.demoAll.control.exceptions.ResourceNotDeletedException;
import com.td3.example.demoAll.model.Hero;
import com.td3.example.demoAll.model.HeroDetails;
import com.td3.example.demoAll.repository.HeroesRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Optional;

@Service
public class HeroesService {
    private HeroesRepository heroesRepository;

    @Autowired
    public HeroesService(HeroesRepository heroesRepository) {
        this.heroesRepository = heroesRepository;
    }

    /* *********************************************
       services to get data
    *********************************************** */
    public List<Hero> findAll() {
        return heroesRepository.findAll();
    }

    public Page<Hero> findAllWithPagination(Pageable pageable) {
        return heroesRepository.findAll(pageable);
    }

    public Hero findHeroById(Long id) throws ResourceException {
        System.out.println("Getting hero by id: " + id);
        Optional<Hero> heroOpt = heroesRepository.findById(id);
        // In case of the hero cannot be found, a HeroNotFoundException is raised. Since it
        // is a sub-subclass of ResourceException, there is no need to modify the header
        // of the current method, by adding throws HeroNotFoundException
        if (heroOpt.isEmpty()) throw new HeroNotFoundException("searching for hero with id = "+id);
        Hero hero = heroOpt.get();
        //System.out.println(hero);
        return hero;
    }

    public Hero findHeroByPublicName(String publicName) throws ResourceException {
        System.out.println("Getting hero by public name: " + publicName);
        Optional<Hero> hero = heroesRepository.findByPublicName(publicName);
        if (hero.isEmpty()) throw new HeroNotFoundException("searching for hero with publicName = "+publicName);
        return hero.get();
    }

    /* *********************************************
       services to create data
    *********************************************** */

    public Hero createHero(String publicName, String realName, Date birthday, String bio, Integer size, Double weight) {
        HeroDetails details = new HeroDetails(birthday, bio, size, weight.intValue());

        Hero hero = new Hero(publicName, realName, details);
        heroesRepository.save(hero);
        return hero;
    }

    /* *********************************************
       services to update data.
       NB: this method only updates names and details, not the safehouse
    *********************************************** */

    public Hero updateHero(Long id, String publicName, String realName, Date birthday, String bio, Integer size, Double weight) {
        Hero hero = findHeroById(id);
        if (publicName != null) hero.setPublicName(publicName);
        if (realName != null) hero.setRealName(realName);
        if (birthday != null) hero.getHeroDetails().setBirthday(birthday);
        if (bio != null) hero.getHeroDetails().setBio(bio);
        if (size != null) hero.getHeroDetails().setSize(size);
        if (weight != null) hero.getHeroDetails().setWeight(weight);
        heroesRepository.save(hero);
        return hero;
    }

    /* *********************************************
       services to delete data
    *********************************************** */
    public void deleteHero(Long id) throws ResourceException {
        Optional<Hero> heroOpt = heroesRepository.findById(id);
        if (heroOpt.isEmpty()) throw new HeroNotFoundException("deleting hero with id = "+id);
        try {
            heroesRepository.deleteById(id);
        }
        catch(Exception e) {
            throw new ResourceNotDeletedException("relationship with another table prevents from deleting hero with id = "+id);
        }
    }
}
