package com.td3.example.demoAll.service;

import com.td3.example.demoAll.control.exceptions.ResourceException;
import com.td3.example.demoAll.control.exceptions.ResourceNotFoundException;
import com.td3.example.demoAll.model.Hero;
import com.td3.example.demoAll.model.Safehouse;
import com.td3.example.demoAll.repository.SafehousesRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Optional;

@Service
public class SafehousesService {
    private SafehousesRepository safehousesRepository;
    private HeroesService heroesService;

    @Autowired
    public SafehousesService(SafehousesRepository safehousesRepository, HeroesService heroesService) {
        this.safehousesRepository = safehousesRepository;
        this.heroesService = heroesService;
    }

    /* *********************************************
       services to get data
    *********************************************** */
    public List<Safehouse> findAll() {
        return safehousesRepository.findAll();
    }

    public Safehouse findSafehouseById(Long id) throws ResourceException {
        Optional<Safehouse> safehouseOpt = safehousesRepository.findById(id);
        if (safehouseOpt.isEmpty()) throw new ResourceNotFoundException("searching for safe house id "+id);
        return safehouseOpt.get();
    }

    public Safehouse findSafehouseByName(String name) throws ResourceException {
        Optional<Safehouse> safehouseOpt = safehousesRepository.findByName(name);
        if (safehouseOpt.isEmpty()) throw new ResourceNotFoundException("searching for safe house named "+name);
        return safehouseOpt.get();
    }

    /* *********************************************
       services to create data
    *********************************************** */
    public Safehouse createSafehouse(String name) {
        Safehouse safehouse = new Safehouse(name);
        safehousesRepository.save(safehouse);
        return safehouse;
    }

    /* *********************************************
       services to update data
    *********************************************** */

    /* NB: despite the fact that it is in heroes table that a foreign key represents the primary key
       of the safehouses table, the way to update a hero's safehouse is to do it from the point of view
       of Safehouse object. Indeed, there is a unidirectional one-to-many relationship between Safehouse and Hero.
       It implies that we cannot "associate" a Safehouse to a Hero, but we can add a Hero to a Safehouse (see List<Hero> attribute).
       Nevertheless, it also implies to get a Hero, thus to use HeroesService to do that.
     */
    public Safehouse addHero(Long safehouseId, Long heroId) throws ResourceException {
        Safehouse safehouse = findSafehouseById(safehouseId);
        Hero hero = heroesService.findHeroById(heroId);
        // if hero already in safehouse do nothing
        if (safehouse.getHeroes().contains(hero)) return safehouse;
        safehouse.addHero(hero);
        safehousesRepository.save(safehouse);
        return safehouse;
    }
}
