package com.td3.example.demoAll.model;

import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.td3.example.demoAll.control.filters.CustomHeroListSerializer;
import jakarta.persistence.*;

import java.util.List;

@Entity
@Table(name="safehouses")
public class Safehouse {

    @Id
    @GeneratedValue(strategy= GenerationType.IDENTITY)
    private Long id;

    // using @Column just to illustrate its possible uses.
    @Column(name="name", nullable = false, unique = true, length = 50)
    private String name;

    /* set a one-to-many unidirectional relationship between Safehouse and Hero.
    A Safehouse may be associated to several Hero, but a Hero has a single Safehouse.

    It implies that heroes table contains a foreign key that corresponds to the primary key of safehouses table.

    The fact that the relation is unidirectional means that we can get the list of heroes associated to
    a given safe house, but we cannot get directly the safehouse for a given hero. For that, we must
    get the hero, get the value of the foreign key safehouse_id, and then get the safehouse with this id.

    Also note that @JoinColumn does not specifies nullable = false, so that a hero can be created without
    an associated safehouse (which is not the case for his details, cf. Hero)
     */
    @OneToMany
    @JoinColumn(name="safehouse_id")
    /* NB on @JsonSerialize
        This annotation allows to specify another way to serialize the following attribute than the defualt one.
        In the present case, instead of serializing whole heroes object, the custom serializer just serializes their id.

        This is one of the possibilities to send back a customized result to the client. Another one is that the controller
        returns an DTO object with only the necessary fields (see Team)
     */
    @JsonSerialize(using = CustomHeroListSerializer.class)
    private List<Hero> heroes;

    public Safehouse(Long id, String name, List<Hero> heroes) {
        this.id = id;
        this.name = name;
        this.heroes = heroes;
    }

    public Safehouse(String name, List<Hero> heroes) {
        this.name = name;
        this.heroes = heroes;
    }

    public Safehouse(String name) {
        this.name = name;
    }

    public Safehouse() {
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public List<Hero> getHeroes() {
        return heroes;
    }

    public void setHeroes(List<Hero> heroes) {
        this.heroes = heroes;
    }

    public void addHero(Hero hero) { heroes.add(hero);}

    @Override
    public String toString() {
        return "Safehouse{" +
                "id=" + id +
                ", name='" + name + '\'' +
                ", heroes=" + heroes +
                '}';
    }
}
