package com.td3.example.demoAll.model;

import jakarta.persistence.*;

@Entity
@Table(name = "powers")
public class Power {

    @Id
    @GeneratedValue(strategy= GenerationType.IDENTITY)
    private Long id;
    
    @Column(name="name", nullable = false, unique = true, length = 50)
    private String name;

    /* NB on powerType attribute:

    A Power is associated to a single PowerType, and a PowerType may be associated to several Power. Thus
    there are two solutions to represent this relation:
        - setting a one-to-many relationship between Power and PowerType.
        - setting a many-to-One relationship between PowerType and Power.

     Note that in both case, it implies to have a foreign key in powers table, which references a primary key in powertypes

    The second solution is chosen because we only need to retrieve the PowerType for a given Power and not the opposite.
    Thus, there is no need to have something like List<Power> powers in PowerType class. This is also why this relationship is unidirectional.
    */
    @ManyToOne(fetch = FetchType.EAGER, optional = false )
    /*
       - no name => name is power_type_id, inferred from powerType + the fact that it is a joined column
       - nullable = false => a power has a mandatory type.
     */
    @JoinColumn(nullable = false)
    private PowerType powerType;

    /* NB on lazy fetching PowerType
    If powerType is not fetch directly from DB, the attribute is not null
    but of class HibernateProxy. This is why, when no special Jackson annotation is used,
    there is an exception that is raised, because Jackson tries to serialize the whole object which is
    not possible, because the proxy has two extra fields compared to the original class: hibernateLazyInitializer & handler.

    In order to mae the serialization successful, we must remove these two fields from the Jackson serialization process by using:
        @JsonIgnoreProperties({"hibernateLazyInitializer", "handler"})

    BUT, even if it works, it will lead to an automatic fetching of powerType in DB, as if an EAGER fetching, when the object
    is returned by a controller, then serialized and sent back to the client.

    In order to avoid this automatic fetch, we can use a custom serialization filter that tests if powerType has been
    loaded or not. this is achieved by the LazyFieldFilter class, and the annotation:
        @JsonInclude(value = JsonInclude.Include.CUSTOM, valueFilter =  LazyFieldFilter.class)

    The whole solution to set this behavior is the following:

    @ManyToOne(fetch = FetchType.LAZY, optional = false, cascade = CascadeType.ALL )
    @JoinColumn(nullable = false) // no name => name is power_type_id, inferred from powerType + the fact that it is a joined column
    @JsonInclude(value = JsonInclude.Include.CUSTOM, valueFilter =  LazyFieldFilter.class)
    @JsonIgnoreProperties({"hibernateLazyInitializer", "handler"})
    private PowerType powerType;
     */


    /* NB: about the interest of setting a bidirectional many-to-one relationship between HeroPowerLevel and Power

    In order to set a bidirectional relationship, we would add here the following :
        @OneToMany(mappedBy = "power")
        @JsonManagedReference
        private List<HeroPowerLevel> heroPowerLevels;

    There also would be a @JsonBackReference in PowerLevel class, as below
        @ManyToOne
        @MapsId("powerId")
        @JoinColumn(name = "power_id")
        @JsonBackReference
        Power power;

        Indeed, @JsonManageReference & @JsonBackReference are mandatory to avoid a recursive serialization:

        Nevertheless, this behavior prevents the serialization of the power attribute. Moreover, there is no need
        to search power levels for a given power (id/name). Thus, this bidirectional is useless
     */


    public Power(Long id, String name, PowerType powerType) {
        this.id = id;
        this.name = name;
        this.powerType = powerType;
    }

    public Power(String name, PowerType powerType) {
        this.name = name;
        this.powerType = powerType;
    }

    public Power() {
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public PowerType getPowerType() {
        return powerType;
    }

    public void setPowerType(PowerType powerType) {
        this.powerType = powerType;
    }

    @Override
    public String toString() {
        return "Power{" +
                "id=" + id +
                ", name='" + name + '\'' +
                ", powerType=" + powerType +
                '}';
    }
}
