package com.td3.example.demoAll.model;

import com.fasterxml.jackson.annotation.JsonManagedReference;
import jakarta.persistence.*;
import org.springframework.lang.NonNull;

import java.util.List;

@Entity
@Table(name = "heroes")
public class Hero {

    @Id
    @GeneratedValue(strategy= GenerationType.IDENTITY)
    private Long id;

    // using @Column just to illustrate its possible uses.
    @Column(name="publicname", nullable = false, unique = true, length = 50)
    private String publicName;
    // BdD name will be by default real_name
    private String realName;

    /* Set a one-to-one (unidirectional) relationship between Hero and HeroDetails.
    There are 2 solutions :
       - 1: heroes table contains a foreign key that corresponds to the primary key of herodetails table.
       - 2: herodetails table contains a foreign key that corresponds to the primary key of heroes table.
    Solution 1 is applied here, for no particular reason.

    Cascading all allows to save a Hero entity without saving first HeroDetails. JPA will do it for us
    Since all is used, it is the same for updating/deleting/...
     */
    @OneToOne(cascade = CascadeType.ALL)
    /* NB one @JoinColumn below:
       - no name given to column for the foreign key => default name is hero_details_id
       - not  nullable => the foreign key cannot be null, which implies taht creating a hero
       cannot be done without creating its details first.
    */
    @JoinColumn(nullable = false)
    private HeroDetails heroDetails;


    /* Set a one-to-many bidirectional relationship between Hero and HeroPowerLevel, so that
       when we get a Hero, spring automatically gets all its power levels.
       This side is the "owned" side, and thus using @OneToMany and mappedBy
       The value of mappedBy must be the attribute name used in HeroPowerLevels to represent a Hero.

       Cascade on remove is mandatory so taht when a hero is deleted, the corresponding rows in
       heroes_powers will be also removed. Otherwise, there would be an integrity violation in that
       table because some keys would still contains a reference to the id of the deleted hero.
    */
    @OneToMany(mappedBy = "hero", cascade = CascadeType.REMOVE)
    /* Since HeroPowerLevel contains a back reference to Hero, a serialization would "by default" lead to
       an infinite recursion: when serializing a Hero, it serialize each HeroPowerLevel, which in return
       serializes the Hero, which serializes each HeroPowerLevel, ...

       To avoid this recursion, there are 2 methods :
          - using @JsonManagedReference and @JsonBackReference
          - not using the default serialization in controllers

        For heroPowerLevels, it is the first solution that is applied.
        For the second solution, see Team & TeamController class
     */
    @JsonManagedReference
    private List<HeroPowerLevel> heroPowerLevels;

    /* NB on the unidirectional many-to-many relationship between Hero and Team:

       It implies that Hero class declares no attribute to represent the list of teams the hero
       belongs to.

       In case of we would like to get bidirectional relationship, we should add:
       @ManyToMany(mappedBy="heroes")
       List<Team> teams;
    */


    public Hero(Long id, String publicName, String realName, List<HeroPowerLevel> heroPowerLevels, HeroDetails heroDetails) {
        this.id = id;
        this.publicName = publicName;
        this.realName = realName;
        this.heroPowerLevels = heroPowerLevels;
        this.heroDetails = heroDetails;
    }

    public Hero(String publicName, String realName, List<HeroPowerLevel> heroPowerLevels, HeroDetails heroDetails) {
        this.publicName = publicName;
        this.realName = realName;
        this.heroPowerLevels = heroPowerLevels;
        this.heroDetails = heroDetails;
    }

    public Hero(String publicName, String realName, HeroDetails heroDetails) {
        this.publicName = publicName;
        this.realName = realName;
        this.heroDetails = heroDetails;
    }

    public Hero() {
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public String getPublicName() {
        return publicName;
    }

    public void setPublicName(String publicName) {
        this.publicName = publicName;
    }

    public String getRealName() {
        return realName;
    }

    public void setRealName(String realName) {
        this.realName = realName;
    }


    public List<HeroPowerLevel> getHeroPowerLevels() {
        return heroPowerLevels;
    }

    public void setHeroPowerLevels(List<HeroPowerLevel> heroPowerLevels) {
        this.heroPowerLevels = heroPowerLevels;
    }

    public HeroDetails getHeroDetails() {
        return heroDetails;
    }

    public void setHeroDetails(HeroDetails heroDetails) {
        this.heroDetails = heroDetails;
    }

    @Override
    public String toString() {
        return "Hero{" +
                "id=" + id +
                ", publicName='" + publicName + '\'' +
                ", realName='" + realName + '\'' +
                ", heroPowerLevels=" + heroPowerLevels +
                ", heroDetails=" + heroDetails +
                '}';
    }
}
