package com.td3.example.demoAll.control;

import com.td3.example.demoAll.control.exceptions.ResourceException;
import com.td3.example.demoAll.control.exceptions.ResourceNotCreatedException;
import com.td3.example.demoAll.model.DTO.HeroPowerLevelDTO;
import com.td3.example.demoAll.model.Hero;
import com.td3.example.demoAll.model.DTO.HeroDTO;
import com.td3.example.demoAll.model.HeroPowerLevel;
import com.td3.example.demoAll.service.HeroesService;
import com.td3.example.demoAll.service.HeroPowerLevelsService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PageableDefault;
import org.springframework.data.web.SortDefault;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
public class HeroesController {

    private final HeroesService heroesService;
    private final HeroPowerLevelsService heroPowerLevelsService;

    @Autowired
    HeroesController(HeroesService heroesService, HeroPowerLevelsService heroPowerLevelsService) {
        this.heroesService = heroesService;
        this.heroPowerLevelsService = heroPowerLevelsService;
    }

    // get all heroes
    @GetMapping("/heroes")
    public List<Hero> getAllHeroes() {
        List<Hero> list = heroesService.findAll();
        return list;
    }
    // get all heroes with pagination. NB: cannot use the same path as without pagination because return value is of different type
    @GetMapping("/heroes/paging")
    public Page<Hero> getAllHeroesWithPagination(@PageableDefault(page = 0, size = 5)
                                                     @SortDefault.SortDefaults({
                                                             @SortDefault(sort = "id", direction = Sort.Direction.ASC)
                                                     })Pageable pageable) {
        return heroesService.findAllWithPagination(pageable);
    }



    // create a hero
    @PostMapping("/heroes")
    public Hero createHero(@RequestBody HeroDTO hero) throws ResourceException {
        // check if body contains hero details, if not => invalid request
        if (hero == null) {
            throw new ResourceNotCreatedException("invalid body while creating a new hero: no HeroDTO provided");
        }
        else if (hero.getHeroDetails() == null) {
            throw new ResourceNotCreatedException("invalid body while creating a new hero: no details on hero provided");
        }
        return heroesService.createHero( hero.getPublicName(), hero.getRealName(), hero.getHeroDetails().getBirthday(), hero.getHeroDetails().getBio(),  hero.getHeroDetails().getSize(),  hero.getHeroDetails().getWeight());
    }

    // update hero (only names & details) by id
    @PutMapping("/heroes")
    public Hero updateHero(@RequestBody HeroDTO hero) throws ResourceException {
        if (hero == null) {
            throw new ResourceNotCreatedException("invalid body while updating a hero: no HeroDTO provided");
        }

        if (hero.getHeroDetails() != null) {
            return heroesService.updateHero(hero.getId(), hero.getPublicName(), hero.getRealName(), hero.getHeroDetails().getBirthday(), hero.getHeroDetails().getBio(), hero.getHeroDetails().getSize(), hero.getHeroDetails().getWeight());
        }
        return heroesService.updateHero(hero.getId(), hero.getPublicName(), hero.getRealName(), null, null, null, null);
    }

    // get hero by id
    @GetMapping("/heroes/{id}")
    public Hero getHeroById(@PathVariable Long id) throws ResourceException {
        Hero hero = heroesService.findHeroById(id);
        return hero;
    }

    @DeleteMapping("/heroes/{id}")
    public void deleteHeroById(@PathVariable Long id) throws ResourceException {
        heroesService.deleteHero(id);
    }

    /* Note on the opportunity to update hero's safehouse from here:
        It is possible but it is generally a very bad idea to have several routes to do the same job.
        Moreover, it would imply to declare & autowire a SafehouseService attribute, and to call its method to add a hero to a safehouse.
        This attribute would be useful just for this single case, so it is not really "efficient" since there exists another way.
     */

    // get hero's powers by id
    @GetMapping("/heroes/{id}/powers")
    public List<HeroPowerLevel> getHeroPowersById(@PathVariable Long id) throws ResourceException {
        List<HeroPowerLevel> list = heroPowerLevelsService.findHeroPowers(id);
        return list;
    }

    // add a power to a hero
    @PatchMapping("/heroes/{id}/powers")
    public List<HeroPowerLevel> addPowerToHero(@PathVariable Long id, @RequestBody HeroPowerLevelDTO heroPowerLevelDTO) throws ResourceException {
        if (heroPowerLevelDTO == null) {
            throw new ResourceNotCreatedException("invalid body while updating hero's powers: no HeroPowerLevelDTO provided");
        }
        List<HeroPowerLevel> list = heroPowerLevelsService.addPowerToHero(id, heroPowerLevelDTO.getPowerId(), heroPowerLevelDTO.getLevel());
        return list;
    }

    @GetMapping("/heroes/getbypublicname/{name}")
    public Hero getHeroByPublicName(@PathVariable String name) throws ResourceException {
        Hero hero = heroesService.findHeroByPublicName(name);
        return hero;
    }
}
