package com.td2.example.demo1.control;

import com.td2.example.demo1.control.exceptions.ResourceNotFoundException;
import com.td2.example.demo1.model.Hero;
import com.td2.example.demo1.model.DTO.HeroDTO;
import com.td2.example.demo1.service.HeroesService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.web.PageableDefault;
import org.springframework.data.web.SortDefault;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
public class HeroesController {

    private final HeroesService heroesService;

    @Autowired
    HeroesController(HeroesService heroesService) {
        this.heroesService = heroesService;
    }

    // get all heroes
    @GetMapping("/heroes")
    public List<Hero> getAllHeroes() {
        List<Hero> list = heroesService.findAll();
        return list;
    }
    // get all heroes with pagination. NB: cannot use the same path as without pagination because return value is of different type
    @GetMapping("/heroes/paging")
    public Page<Hero> getAllHeroesWithPagination(@PageableDefault(page = 0, size = 5)
                                                     @SortDefault.SortDefaults({
                                                             @SortDefault(sort = "id", direction = Sort.Direction.ASC)
                                                     })Pageable pageable) {
        return heroesService.findAllWithPagination(pageable);
    }

    // create a hero
    @PostMapping("/heroes")
    public Hero createHero(@RequestBody HeroDTO hero)  {
        return heroesService.createHero( hero.getPublicName(), hero.getRealName(), hero.getPower(), hero.getPowerLevel());
    }

    // update hero by id
    @PutMapping("/heroes")
    public Hero updateHero(@RequestBody HeroDTO hero) throws ResourceNotFoundException {
        return heroesService.updateHero(hero.getId(), hero.getPublicName(), hero.getRealName(), hero.getPower(), hero.getPowerLevel());
    }

    // get hero by id
    @GetMapping("/heroes/{id}")
    public Hero getHeroById(@PathVariable Long id) throws ResourceNotFoundException {
        Hero hero = heroesService.findHeroById(id);
        return hero;
    }

    // delete hero by id
    @DeleteMapping("/heroes/{id}")
    public void deleteHeroById(@PathVariable Long id) throws ResourceNotFoundException {
        heroesService.deleteHero(id);
    }

    @GetMapping("/heroes/getbypublicname/{name}")
    public Hero getHeroByPublicName(@PathVariable String name) throws ResourceNotFoundException {
        Hero hero = heroesService.findHeroByPublicName(name);
        return hero;
    }

    // example with response manipulation
    @GetMapping("/heroes/getbypower")
    public ResponseEntity<List<Hero>> getHeroesByPower(@RequestParam String power, @RequestParam(required = false) Boolean pattern) {
        List<Hero> list = null;
        if (pattern == null) pattern = false;
        list = heroesService.findAllByPower(power,pattern);

        // if list is empty, do not send the empty list but instead a NO_CONTENT status code
        if (list.isEmpty()) return ResponseEntity.status(HttpStatus.NO_CONTENT).body(null);
        // if list is not empty, send it, together with 2 customs response headers
        return ResponseEntity
                .ok()
                .header("myheader","toto")
                .header("otherheader","hello")
                .body(list);
    }

    @GetMapping("/heroes/count")
    public long getItemsCount() {
        long count = heroesService.count();
        return count;
    }
}
