package com.td2.example.demo2.service;

import com.td2.example.demo2.control.exceptions.HeroNotFoundException;
import com.td2.example.demo2.control.exceptions.ResourceNotFoundException;
import com.td2.example.demo2.model.Hero;
import com.td2.example.demo2.repository.HeroesRepository;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Optional;

@Service
public class HeroesService {
    private HeroesRepository heroesRepository;

    @Autowired
    public HeroesService(HeroesRepository heroesRepository) {
        this.heroesRepository = heroesRepository;
    }

    /* *********************************************
       services to get data
    *********************************************** */
    public List<Hero> findAll() {
        return heroesRepository.findAll();
    }

    public Page<Hero> findAllWithPagination(Pageable pageable) {
        return heroesRepository.findAll(pageable);
    }

    public Hero findHeroById(Long id) throws ResourceNotFoundException {
        System.out.println("Getting hero by id: " + id);
        Optional<Hero> hero = heroesRepository.findById(id);
        // In case of the hero cannot be found, a HeroNotFoundException is raised. Since it
        // is a subclass of ResourceNotFoundException, there is no need to modify the header
        // of the current method, by adding throws HeroNotFoundException
        if (hero.isEmpty()) throw new HeroNotFoundException("searching for hero with id = "+id);
        return hero.get();
    }

    public Hero findHeroByPublicName(String publicName) throws ResourceNotFoundException {
        System.out.println("Getting hero by public name: " + publicName);
        Optional<Hero> hero = heroesRepository.findByPublicName(publicName);
        // Normally, this method would throw a HeroNotFoundException. But in order to show how errors are
        // managed by the ResourceNotFoundAdvice class, a more general ResourceNotFoundException is thrown
        if (hero.isEmpty()) throw new ResourceNotFoundException("searching for hero with publicName = "+publicName);
        return hero.get();
    }

    public List<Hero> findAllByPower(String string, boolean pattern) {
        List<Hero> list = null;
        if (pattern) {
            System.out.println("Getting heroes with power containing " + string);
            list = heroesRepository.findAllByPowerContaining(string);
        }
        else {
            System.out.println("Getting heroes with power = " + string);
            list = heroesRepository.findAllByPower(string);
        }
        return list;
    }

    public long count() {
        long count = heroesRepository.count();
        return count;
    }

    /* *********************************************
       services to create data
    *********************************************** */
    public Hero createHero(String publicName, String realName, String power, Integer powerLevel) {
        // NB: realName, power and powerLeve may be null because no nullable = false is used the Hero model
        Hero hero = new Hero(publicName, realName, power, powerLevel);
        heroesRepository.save(hero);

        return hero;
    }
    /* *********************************************
       services to update data
    *********************************************** */
    public Hero updateHero(Long id, String publicName, String realName, String power, Integer powerLevel) throws ResourceNotFoundException {
        Optional<Hero> heroOpt = heroesRepository.findById(id);
        if (heroOpt.isEmpty()) throw new HeroNotFoundException("updating hero with id = "+id);

        Hero hero = heroOpt.get();
        if (publicName != null) hero.setPublicName(publicName);
        if (realName != null) hero.setRealName(realName);
        if (power != null) hero.setPower(power);
        if (powerLevel != null) hero.setPowerLevel(powerLevel);
        heroesRepository.save(hero);
        return hero;
    }

    /* *********************************************
       services to delete data
    *********************************************** */
    public void deleteHero(Long id) {
        Optional<Hero> heroOpt = heroesRepository.findById(id);
        if (heroOpt.isEmpty()) throw new HeroNotFoundException("deleting hero with id = "+id);
        heroesRepository.deleteById(id);
    }
}
