package com.td2.example.demo2.service;

import com.td2.example.demo2.control.exceptions.CSVConversionException;
import com.td2.example.demo2.model.Hero;
import com.td2.example.demo2.repository.HeroesRepository;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

@Service
public class CSVService {

    private HeroesRepository heroesRepository;

    @Autowired
    public CSVService(HeroesRepository heroesRepository) {
        this.heroesRepository = heroesRepository;
    }

    // private method to parse any csv from an input stream, used by conversion method aas a preliminary
    private Iterable<CSVRecord> parseCSV(InputStream is) throws CSVConversionException {

        try {
            BufferedReader fileReader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
            CSVParser csvParser = new CSVParser(fileReader,
                    CSVFormat.DEFAULT.withFirstRecordAsHeader().withIgnoreHeaderCase().withTrim());
            Iterable<CSVRecord> csvRecords = csvParser.getRecords();
            return csvRecords;
        }
        catch (IOException e) {
            throw new CSVConversionException("fail to read from CSV file: " + e.getMessage());
        }
    }

    // save heroes in DB from a multipart file, e.g. an uploaded file
    public void saveCSVToHeroes(MultipartFile file) throws CSVConversionException {
        if (! "text/csv".equals(file.getContentType())) {
            throw new CSVConversionException("provided file is not a csv");
        }
        saveCSVToHeroes(file.getResource());
    }

    // save heroes in DB from a resource file
    public void saveCSVToHeroes(Resource resource) throws CSVConversionException {

        try {
            Iterable<CSVRecord> csvRecords = parseCSV(resource.getInputStream());
            List<Hero> heroes = convertCSVToHeroes(csvRecords);
            heroesRepository.saveAll(heroes);
        }
        catch(IOException e) {
            throw new CSVConversionException("cannot read provided csv file");
        }
    }

    // conversion from a list of csv records to a list of heroes
    private List<Hero> convertCSVToHeroes(Iterable<CSVRecord> csvRecords) throws CSVConversionException {
        try {
            List<Hero> heroes = new ArrayList<>();
            for (CSVRecord csvRecord : csvRecords) {
                Hero hero = new Hero(
                        csvRecord.get("publicName"),
                        csvRecord.get("realName"),
                        csvRecord.get("power"),
                        Integer.parseInt(csvRecord.get("powerLevel"))
                );

                heroes.add(hero);
            }
            return heroes;
        }
        catch(IllegalStateException e) {
            throw new CSVConversionException("invalid CSV file: no headers provided");
        }
        catch(IllegalArgumentException e) {
            throw new CSVConversionException("fail to get value from CSV file: " + e.getMessage());
        }
    }
}
