import java.io.*;
import java.net.*;

class ServerThreadDiamond extends Thread  {

    private Socket sockComm;
    private PrintStream ps;
    private BufferedReader br;
    private int idPseudo; // the id of my pseudo in the list managed by party
    private int idPlayer; // the unique identifier of the client and thus the thread.
    private Party party;
    private int idThread;
    private String pseudo;

    public ServerThreadDiamond(Socket sockComm, Party party, int idThread) {
        this.sockComm = sockComm;
        this.party = party;
        this.idThread = idThread;
        idPlayer = -1;
        idPseudo = -1;
        pseudo = null;
    }

    public void run() {

        try {
            ps = new PrintStream(sockComm.getOutputStream());
            br = new BufferedReader(new InputStreamReader(sockComm.getInputStream()));
        }
        catch(IOException e) {
            System.err.println("Cannot initialize streams. Aborting");
            return;
        }

        // NB : until party is really started, there may be more than 2 threads that are
        // competing to participate to the party. It is only when trying to register a pseudo
        // that the thread will know if it can really participate or have to disconnect because
        // to other threads are already taken.

        // check if client is authorized to send a pseudo
        if (party.getState() == Party.PARTY_STATE_PLAYING) {
            // tell client to stop connection
            ps.println("ERR PARTY_STARTED");
            return;
        }
        else {
            // tell client to keep going with pseudo setup
            ps.println("OK");
        }

        // obtain the pseudo from the client
        if (!getPseudo()) {
            System.out.println("I/O error while setting the pseudo. Aborting");
            return;
        }

        /* Now, try to declare the client as a valid player by registering its pseudo in the party
           Note that a thread A may be interrupted between getPseudo() and setPseudo(). So there is a chance
           that another thread B calls setPseudo() before A, taking its place as a party player.
           This is why setPseudo() must check once again if the pseudo does not exists AND if there
           are less than two registered pseudo
        */
        if ((idPlayer = party.setPseudo(pseudo)) >= 0) {
            // tell client to keep going with the party
                /* INFO:
                Even if println() fails because client disconnected just before, there will be no
                exception thrown (cf. PrintStream class). Thus, we are sure that the current thread will reach the waitPartyStart()
                and if it is the second thread to reach it, it will unblock the first thread.
                 */
            ps.println("OK");
        }
        else {
            // tell client to disconnect
            ps.println("ERR PSEUDO_REFUSED");
            return;
        }

        // waiting for two clients, in order to start the party
        party.waitPartyStart();

        // now play
        partyLoop();

        System.out.println("Thread ["+idThread+"] - waiting for party end.");
        // waiting for the two thread to finish the party, normally or suddenly
        // only the second resets the party object
        party.waitPartyEnd();
        try {
            sockComm.close();
        }
        catch(IOException e) {}
        System.out.println("Thread ["+idThread+"] - stop.");
    }


    public boolean getPseudo() {

        boolean stop = false;
        System.out.println("Thread ["+idThread+"] - entering initLoop().");
        while(!stop) {
            try {
                pseudo = br.readLine();
            }
            catch(IOException e) {
                return false;
            }
            // recv null means broken connection => stop thread by returning false
            if (pseudo == null) return false;
            if (!party.isPseudoExists(pseudo)) {
                ps.println("OK");
                stop = true;
            }
            else {
                ps.println("ERR PSEUDO");
            }
        }
        // at this point, pseudo is known and valid
        return true;
    }

    public void partyLoop() {
        String line = null;
        System.out.println("Thread ["+idThread+"] - entering partyLoop().");

        int pawnValue = -1;

        // send the player id to the client
        ps.println(idPlayer);

        int id = -1;
        byte value = -1;
        // both players are playing 6 times, so there are 12 turns in a party
        for(int i=0;i<12;i++) {
            // 0 - check the state of the party. In case of other thread has ended, stop the thread
            if (party.getState() == Party.PARTY_STATE_END)  return;
            // 1 - send board
            ps.println(party.getBoard());

            // 2 - send the id of the current player
            ps.println(party.getCurrentPlayer());

            // 3 - If I am the thread of the current player
            if (idPlayer == party.getCurrentPlayer()) {
                pawnValue = party.getPawnValue(i/2);
                // send value & get what to play, until it is valid
                if (!getOrder(pawnValue)) {
                    // getting false means client disconnected => end the party
                    return;
                }
                // set the next player
                party.setNextPlayer();
                // If I'm the last player of the party, I can directly compute the score
                if (i==11) party.computePartyResult();
            }

            /* BEWARE - dead-lock case :
                - thread 0 is NOT the current player and is already stuck in waitCurrentPlayed()
                - thread 1 IS the current player and getOrder() returns false

                => thread 1 quit partyLoop(), goes into waitPartyEnd() and waits for thread 0,
                which is a dead-lock.

                Solution: in such case, thread 1 MUST unblock thread 0 by put tokens in semCurrentPlayed()
                before. This can be done different ways but the simplest is to do this within waitPartyEnd()
                when the first thread to stop enters in this method.
             */
            party.waitCurrentPlayed();

            // if there is still turns to play
            if (i < 11) {
                ps.println("CONT");
            }
        }
        // abnormal end of the party
        if (party.getState() == Party.PARTY_STATE_END) return;

        // normal end of the party
        ps.println("END "+party.getPartyScore());
        ps.println(party.getBoard());
    }

    public boolean getOrder(int pawnValue) {
        boolean valid = false;
        String line = null;
        // send the pawn value to the client
        ps.println(pawnValue);
        //read what to play
        while (!valid) {
            try {
                line = br.readLine();
            }
            catch(IOException e) {
                return false;
            }
            // recv null means broken connection => stop thread by throwing exception
            if (line == null) return false;
            int idCell = -1;
            try {
                idCell = Integer.parseInt(line);
            }
            catch (NumberFormatException e) {
                ps.println("ERR invalid cell number");
            }
            valid = party.testAndPlay(idCell, idPlayer*6+pawnValue);
            if (valid) {
                ps.println("OK");
            }
            else {
                ps.println("ERR invalid cell number");
            }
        }
        return true;
    }
}
