import java.util.*;

public class Party {

    private static Random loto = new Random(Calendar.getInstance().getTimeInMillis());

    public static int PARTY_STATE_IDLE = 0;
    public static int PARTY_STATE_PLAYING = 1;
    public static int PARTY_STATE_END = 2;

    int state; // state of the party : 0 = wait to start, 1 = playing, 2 = end

    Board board;
    int currentPlayer; // when state is playing, gives the current player's id : 0 or 1, alternating
    List<String> playersPseudo;

    Semaphore semPartyStart; // semaphore to wait the beginning of the party, using nbPartyStart as a counter
    int nbPartyStart; // number of threads waiting for the party to start
    Semaphore semCurrentPlayed; // semaphore to wait that current player played sthg. CAUTION: the current thread MUST also use this semaphore
    int nbCurrentPlayed; // number of thread that are waiting for current player played
    Semaphore semEndParty; // semaphore to wait the end of the party
    int nbPartyEnd;

    List<Integer> pawnValuesPlayer0; // same but for the pleyr
    List<Integer> pawnValuesPlayer1; // same but for the pleyr

    public Party()  {
        // just create what must be created a single time
        // because reset() will be called to initialize attributes values
        board = new Board();
        semPartyStart = new Semaphore(0);
        semEndParty = new Semaphore(0);
        playersPseudo = new ArrayList<>();
        pawnValuesPlayer0 = new ArrayList<>();
        pawnValuesPlayer1 = new ArrayList<>();

        reset();
    }


    // reset() is only called by the main server before player's thread are created => not synchronized
    public void reset() {
        board.clearBoard();
        state = PARTY_STATE_IDLE;

        playersPseudo.clear();

        nbPartyStart = 0;
        nbPartyEnd = 0;

        // recreate semaphore to be sure there are no tokens in it.
        semCurrentPlayed = new Semaphore(0);
        nbCurrentPlayed = 0;

        currentPlayer = -1;
        pawnValuesPlayer0.clear();
        pawnValuesPlayer1.clear();
        for(int i=1;i<=6;i++) {
            pawnValuesPlayer0.add(i);
            pawnValuesPlayer1.add(i);
        }
        Collections.shuffle(pawnValuesPlayer0);
        Collections.shuffle(pawnValuesPlayer1);
    }

    public synchronized String getBoard() {
        return board.toString();
    }

    public synchronized int getCurrentPlayer() {
        return currentPlayer;
    }

    public synchronized int getState() {
        return  state;
    }

    /*
    get the pawn value that must be played by a player during its turn #playerTurn
     */
    public synchronized int getPawnValue(int playerTurn) {
        if (currentPlayer == 0) {
            return pawnValuesPlayer0.get(playerTurn);
        }
        else if (currentPlayer == 1) {
            return pawnValuesPlayer1.get(playerTurn);
        }
        return -1;
    }

    public synchronized String getPseudo(int index) {
        return playersPseudo.get(index);
    }

    // returns true if pseudo already exists, false otherwise
    public synchronized boolean isPseudoExists(String pseudo) {
        return playersPseudo.contains(pseudo);
    }

    public synchronized int setPseudo(String pseudo) {
        if ((playersPseudo.contains(pseudo)) || (playersPseudo.size() == 2)) {
            return -1;
        }
        playersPseudo.add(pseudo);
        return playersPseudo.size()-1;
    }

    public void waitPartyStart() {
        nbPartyStart += 1;
        if (nbPartyStart == 2) {
            // reset to 0 for the next party
            nbPartyStart = 0;
            // 0 is the first player
            currentPlayer = 0;
            state = PARTY_STATE_PLAYING;
            semPartyStart.put(2); // put 2 tokens so that both threads can stop waiting
        }
        semPartyStart.get(1); // get one token
    }

    public synchronized boolean testAndPlay(int idCell, int value) {
        if (board.isCellEmpty(idCell)) {
            board.setPawn(idCell,(byte)value);
            return true;
        }
        return false;
    }

    /*
        WARNING: the current thread MUST also call this method so that 2 tokens are put in the semaphore
        and both thread can pass to the next turn   
     */
    public void waitCurrentPlayed() {
        nbCurrentPlayed += 1;
        if (nbCurrentPlayed == 2) {
            nbCurrentPlayed = 0;
            semCurrentPlayed.put(2);
        }
        semCurrentPlayed.get(1);
    }

    public synchronized void setNextPlayer() {
        currentPlayer = (currentPlayer+1)%2;
    }

    public synchronized void computePartyResult() {
        board.computeScore();
    }

    public synchronized String getPartyScore() {
        return board.getBlueScore()+ " " + board.getRedScore();
    }

    public void waitPartyEnd() {
        state = PARTY_STATE_END;
        // put tokens in semCurrentPlayed in case of the other thread is stuck
        // since this semaphore is recreated at each party, it has no side effect if
        // tokens are in fact useless.
        semCurrentPlayed.put(2);
        nbPartyEnd += 1;
        if (nbPartyEnd == 2) {
            reset();
            semPartyStart.put(2); // put 2 tokens so that both threads can stop waiting
        }
        semPartyStart.get(1); // get one token
    }
}
