/**
 * defines methods to interact with Hero documents
 * @module HeroController
 */
const Organization = require('../../models/herocorp/organization.model')
const OrganizationErrors = require('../../commons/organization.errors')
const Team = require('../../models/herocorp/team.model');
const TeamErrors = require('../../commons/team.errors')
const Helpers = require('./helpers.controller')

const {answer} =  require('../ControllerAnswer')

const create = async function (req, res, next) {
  answer.reset()
  console.log('create org with req.body = '+JSON.stringify(req.body));
  // sanity check on required parameters
  if ( (!req.body.name) || (!req.body.secret) ) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_CREATE_INVALID_DATA))
    return next(answer);
  }
  // search if name already exists (nb: syntax using exec(), await and try/catch)
  try {
    let org = await Organization.findOne({ $or: [{name: req.body.name},{secret: req.body.secret}] } ).exec()
    if (org !== null) {
      answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_CREATE_ALREADY_EXISTS))
      return next(answer);
    }
  }
  catch(err) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_CREATE_FAILED))
    return next(answer);
  }

  // make an object with required fields
  let o = {
    name: req.body.name,
    secret: req.body.secret,
    teams: []
  };

  //insert the hero in DB (nb: create() uses callbacks and not await/exec)
  Organization.create(o, function(err, org) {
    if (err) {
      answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_CREATE_FAILED))
      return next(answer);
    }
    if (org === null) {
      answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_CREATE_FAILED))
      return next(answer);
    }
    else {
      // sends back the whole org
      answer.setPayload(org)
      res.status(201).send(answer);
    }
  });
};

const update = async function (req, res, next) {
  answer.reset()
  console.log('update an organization');

  answer.setPayload("update of an org not implemented yet.")
  res.status(200).send(answer);
};

/*
  req.body.team is the _id of a team
  NB: the org is found from the org-secret header of query param
 */
const addTeam = async function (req, res, next) {
  answer.reset()
  console.log('addHeroes team to an organization');
  if (!req.body.idTeam) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_ADDTEAM_INVALID_DATA))
    return next(answer);
  }
  await Helpers.checkOrgSecret(req)
  if (answer.isError()) {
    return next(answer)
  }
  let org = answer.getPayload()
  // check if _id in req.body._id exists. If not => error
  let team = null
  try {
    team = await Team.findOne({_id: req.body.idTeam}).exec()
    if (team === null) {
      answer.set(TeamErrors.getError(TeamErrors.ERR_TEAM_GET_FAILED))
      return next(answer);
    }
  } catch (err) {
    answer.set(TeamErrors.getError(TeamErrors.ERR_TEAM_GET_FAILED))
    return next(answer);
  }

  org.teams.push(team._id)
  try {
    org = await org.save()
  }
  catch(err) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_ADDTEAM_FAILED))
    return next(answer);
  }
  // update each hero to be sure that the org. is present in the organizations field

  answer.setPayload(org)
  res.status(200).send(answer);
};

/*
  req.body.heroes is an array of _id
  req.body.idOrg is the _id of the affiliation org.
 */
const removeTeam = async function (req, res, next) {
  answer.reset()
  console.log('removeTeam team from an organization');
  // check if all _id in req.body.heroes exist. If not => error
  if (!req.body.idTeam) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_REMTEAM_INVALID_DATA))
    return next(answer);
  }
  await Helpers.checkOrgSecret(req)
  if (answer.isError()) {
    return next(answer)
  }
  let org = answer.getPayload()
  // check if _id in req.body._id exists. If not => error
  let idx = org.teams.findIndex(t => t._id.toString() === req.body.idTeam)
  if (idx === -1) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_REMTEAM_FAILED))
    return next(answer);
  }
  org.teams.splice(idx, 1)
  try {
    org = await org.save()
  }
  catch(err) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_REMTEAM_FAILED))
    return next(answer);
  }
  // update each hero to be sure that the org. is present in the organizations field

  answer.setPayload(org)
  res.status(200).send(answer);
};

const getById = async function (req, res, next) {
  answer.reset()
  console.log('get one organization with id: '+req.params.id);
  if (!req.params.id) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_GET_INVALID_DATA))
    return next(answer);
  }
  // check if a valid secret is provided
  await Helpers.checkOrgSecret(req)
  if (answer.isError()) {
    return next(answer)
  }
  let org = answer.getPayload()
  // now check if org found by secret corresponds to the _id given in params
  if (req.params.id !== org._id.toString()) {
    console.log("org ids do not match")
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_GET_FAILED))
    return next(answer);
  }

  try {
    // get the org. and populate team array so that it contains heroes subdocuments instead
    // of just their _id.
    //org = await Organization.find({_id: req.params.id}).populate({path:'teams', populate: {path: 'members'}}).exec()
    org = await Organization.find({_id: req.params.id}).populate('teams').exec()
  }
  catch(err) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_GET_FAILED))
    return next(answer);
  }

  answer.setPayload(org)
  res.status(200).send(answer);
};

const getAll = async function (req, res, next) {
  answer.reset()
  console.log('get all organizations');
  // search if names already exists (nb: syntax using exec(), await and try/catch)
  let list = null
  try {
    list  = await Organization.find({}, '_id name').exec()
  }
  catch(err) {
    answer.set(OrganizationErrors.getError(OrganizationErrors.ERR_ORGANIZATION_GETALL_FAILED))
    return next(answer);
  }
  answer.setPayload(list)
  res.status(200).send(answer);
};

module.exports = {
  create,
  update,
  addTeam,
  removeTeam,
  getById,
  getAll,
};
