/**
 * defines methods to interact with Hero documents
 * @module HeroController
 */
const Hero = require('../../models/herocorp/heroe.model');
const HeroErrors = require('../../commons/hero.errors')
const Helpers = require('./helpers.controller')

const {answer} =  require('../ControllerAnswer')

/* NB: messages returned by the API have the 2 formats
 - { error: error_number, status: HTTP_request_status, data: error_message } : error_message is a string
 - { error: 0, data: answer_to_request } : answer_to_request is generally a json object containing what is asked
 by the client.

 */

const create = async function (req, res, next) {
  answer.reset()
  console.log('create hero with req.body = '+JSON.stringify(req.body));
  // sanity check on required parameters
  if ((!req.body.publicName) || (!req.body.realName)) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_CREATE_INVALID_DATA))
    return next(answer);
  }

  // search if names already exists (nb: syntax using exec(), await and try/catch)
  try {
    let hh = await Hero.findOne({publicName: req.body.publicName}).exec()
    if (hh !== null) {
      answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_CREATE_ALREADY_EXISTS))
      return next(answer);
    }
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_CREATE_FAILED))
    return next(answer);
  }

  // make an object with required fields
  let h = {
    publicName: req.body.publicName,
    realName: req.body.realName,
  };
  // add mandatory fields if they are provided
  if (req.body.powers) {
    h.powers = req.body.powers;
  }
  else {
    h.powers = []
  }

  //insert the hero in DB (nb: create() uses callbacks and not await/exec)
  let hero = null
  try {
    hero = await Hero.create(h)
    if (hero === null) {
      answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_CREATE_FAILED))
      return next(answer);
    }
    else {
      // sends back the whole hero
      answer.setPayload( hero)
      res.status(201).send(answer);
    }
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_CREATE_FAILED))
    return next(answer);
  }
}

/*
NB: req.body.data contains updated fields of a hero.
   since we use set(), it is not mandatory that req.body.data
   contains all fields defined in the schema. There may be only
   the one that must be updated.
   Nevertheless, there must be at least _id in order to find which
   hero to update
 */
const update = async function (req, res, next) {
  answer.reset()
  console.log('update hero with req.body = '+JSON.stringify(req.body));
  // sanity check on required parameters
  if (!req.body._id) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_INVALID_DATA))
    return next(answer);
  }
  // a hero can be updated only if it is member
  await Helpers.checkAccessToHero(req, req.body._id)
  if (answer.isError()) {
    return next(answer);
  }

  // Since data contains the whole hero, we can find its _id
  // search if names already exists (nb: syntax using exec(), await and try/catch)
  let hero = null
  try {
    hero = await Hero.findOne({_id: req.body._id}).exec()
    if (hero === null) {
      answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_INVALID_DATA))
      return next(answer);
    }
    // now update h
    hero.set(req.body)
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_FAILED))
    return next(answer);
  }
  // now save the hero in DB
  try {
    hero = await hero.save()
    // sends back the whole hero
    answer.setPayload(hero);
    res.status(200).send(answer);
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_FAILED))
    return next(answer);
  }
};

/*
authUpdate is protected by verifyToken, to check if the user
associated to this hero is logged in.
 */
const authUpdate = async function (req, res, next) {
  answer.reset()
  console.log('update auth. hero with req.body = '+JSON.stringify(req.body));
  // sanity check on required parameters
  if (!req.body._id) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_INVALID_DATA))
    return next(answer);
  }
    // Since data contains the whole hero, we can find its _id
  // search if names already exists (nb: syntax using exec(), await and try/catch)
  let hero = null
  try {
    hero = await Hero.findOne({_id: req.body._id}).exec()
    if (hero === null) {
      answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_INVALID_DATA))
      return next(answer);
    }
    // now update h
    hero.set(req.body)
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_FAILED))
    return next(answer);
  }
  // now save the hero in DB
  try {
    hero = await hero.save()
    // sends back the whole hero
    answer.setPayload(hero)
    res.status(200).send(answer);
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_UPDATE_FAILED))
    return next(answer);
  }
};

/**
   Retrieve all infos on a hero, only if the secret provided in header org-secret
   corresponds to an organization, and the hero is in a team that is hosted by the org.
 */
const getById = async function (req, res, next) {
  answer.reset()
  console.log('get one hero with id: '+req.params.id);
  // search if names already exists (nb: syntax using exec(), await and try/catch)

  await Helpers.checkAccessToHero(req, req.params.id)
  console.log(answer)
  if (answer.isError()) {
    return next(answer);
  }

  let hero = null
  try {
    hero = await Hero.find({_id: req.params.id}).exec()
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_GET_FAILED))
    return next(answer);
  }
  answer.setPayload(hero)
  res.status(200).send(answer);
};

const getAliases = async function (req, res, next) {
  answer.reset()
  console.log('get all heroes aliases + their id');
  // search if names already exists (nb: syntax using exec(), await and try/catch)
  let list = null
  try {
    list  = await Hero.find({},'_id publicName').exec()
    console.log(list)
  }
  catch(err) {
    answer.set(HeroErrors.getError(HeroErrors.ERR_HERO_GETALL_FAILED))
    return next(answer);
  }
  answer.setPayload(list)
  res.status(200).send(answer);
};

module.exports = {
  create,
  update,
  authUpdate,
  getById,
  getAliases,
}
