const jwt = require("jsonwebtoken");
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
const { v4: uuidv4 } = require('uuid');


const Config = require("../../commons/config");
const User = require('../../models/auth/user.model');
const RefreshToken = require('../../models/auth/refreshtoken.model');
const UserErrors = require('../../commons/user.errors');
const AuthErrors = require('../../commons/auth.errors');

const {answer} = require('../ControllerAnswer')

/**
 * check user credentials
 * @param {Object} req - The request object (provided by express)
 * @param {Object} req.body - The data payload sent with the request
 * @param {string} req.body.login - The login of the user to sign in
 * @param {string} req.body.password - The password of the user to sign in
 * @param {Object} res - The result object used to send the result to the client (provided by express)
 * @param {Function} next - The next middleware to call after this one
 * @alias module:AuthController.signIn
 */
const signIn = async function (req, res, next) {
  answer.reset()
  console.log('sign in');

  // sanity check on parameters
  if (!req.body.login) {
    answer.set ( AuthErrors.getError(AuthErrors.ERR_AUTH_LOGIN_NOT_DEFINED));
    return next(answer);
  }
  if (!req.body.password) {
    answer.set ( AuthErrors.getError(AuthErrors.ERR_AUTH_PASSWORD_NOT_DEFINED));
    return next(answer);
  }
  // check if name exists
  let user = null;
  try {
    user = await User.findOne({login:req.body.login}).exec();
    if (user === null) {
      answer.set (UserErrors.getError(UserErrors.ERR_USER_CANNOT_FIND_LOGIN));
      return next(answer);
    }
  }
  catch(err) {
    answer.set (UserErrors.getError(UserErrors.ERR_USER_INVALID_FIND_LOGIN_REQUEST));
    return next(answer);
  }
  let passValid = bcrypt.compareSync(req.body.password, user.password);
  if (!passValid) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_PASSWORD_NO_MATCH));
    return next(answer);
  }

  // create an xsrf token that must be stored in local storage/app.store on client side
  const xsrfToken = crypto.randomBytes(64).toString('hex');
  console.log("xsrf token = "+xsrfToken);

  // create the jwt token that it sent as a cookie. It contains a copy of the xsrf token
  let jwtToken = jwt.sign({ id: user._id.toString(), xsrfToken }, Config.secretKey, {
    expiresIn: Config.jwtExpiration
  });
  console.log("jwt token = "+jwtToken);

  // create the refresh token for the jwt token
  await createRefreshToken(user);
  if (answer.isError()) {
    return next(answer);
  }
  console.log(answer)
  let refreshToken = answer.getPayload().token;

  res.cookie('access_token', jwtToken, {
    maxAge: Config.jwtExpiration*1000,
    httpOnly: true,
    secure: true,
    signed: true,
    sameSite: 'none'
  });

  answer.reset()
  answer.setPayload({
    name: user.login,
    xsrfToken: xsrfToken,
    refreshToken: refreshToken,
  });
  console.log(answer);

  res.status(200).send(answer);
};

const createRefreshToken = async function (user) {
  answer.reset()

  let expiredAt = new Date();
  try {
    await RefreshToken.deleteMany({expiryDate: { $lt : expiredAt.getTime()}}).exec();
  }
  catch(err) {
    console.log("error while cleaning session refreshtoken table")
  }

  expiredAt.setSeconds(
      expiredAt.getSeconds() + Config.refreshExpiration
  );

  let tokenStr = uuidv4();
  let refToken = new RefreshToken({
    token: tokenStr,
    user: user._id,
    expiryDate: expiredAt.getTime(),
  })
  try {
    await refToken.save();
    answer.setPayload(refToken);
  }
  catch(err) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_CANNOT_CREATE_REFRESHTOKEN));
    answer.setPayload(answer.getPayload() + '\n' + err);
  }
};
/**
 * check if a jwt token is provided in request header and check it against
 * the secret key
 * @param {Object} req - The request object (provided by express)
 * @param {Object} res - The result object used to send the result to the client (provided by express)
 * @param {Function} next - The next middleware to call after this one
 * @alias module:AuthController.verifyToken
 */
const verifyToken = async function (req, res, next) {

  answer.reset()

  let accessToken = req.signedCookies.access_token;
  console.log(JSON.stringify(req.signedCookies));
  let xsrfToken = req.headers["x-xsrf-token"];

  console.log("verify tokens: jwt="+accessToken+" | xsrf="+xsrfToken);
  // first check if xsrf token exists, and if not it is assumed that no login was sucessful
  if (!xsrfToken) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NO_XSRF_TOKEN));
    return next(answer);
  }

  if (!accessToken) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NO_JWT_TOKEN));
    return next(answer);
  }

  let decoded = '';
  try {
    decoded = jwt.verify(accessToken, Config.secretKey)
  }
  catch(err) {
    if (err instanceof jwt.TokenExpiredError) {
      answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_TOKEN_EXPIRED));
      return next(answer);
    } else {
      answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NOT_AUTHORIZED));
      return next(answer);
    }
  }
  // check if xsrf in header and jwt match
  if (xsrfToken !== decoded.xsrfToken) {
    console.log("decoded and header xsrf tokens differ !")
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NOT_AUTHORIZED));
    return next(answer);
  }
  // now find if user in in DB and bind it the req
  try {
    let user = await User.findById(decoded.id).exec()
    if (user === null) {
      answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NOT_AUTHORIZED));
      return next(answer);
    }
    req.user = user;
  }
  catch(err) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NOT_AUTHORIZED));
    return next(answer);
  }
  return next();
};

const refreshToken = async function (req, res, next) {
  answer.reset()
  let reqToken = req.body.refreshToken;
  let xsrfToken = req.headers["x-xsrf-token"];

  console.log('trying to refresh jwt token using :' +JSON.stringify(reqToken));

  if (!reqToken) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_NO_REFRESH_TOKEN));
    return next(answer);
  }

  let refreshToken = null
  try {
    refreshToken = await RefreshToken.findOne({token: reqToken}).populate('user').exec();
  }
  catch (err) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_INVALID_FIND_REFRESHTOKEN_REQUEST));
    return next(answer);
  }
  // cannot find the token => error
  if (!refreshToken) {
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_REFRESH_NO_MATCH));
    return next(answer);
  }

  if (refreshToken.expiryDate.getTime() < new Date().getTime()) {

    await RefreshToken.findByIdAndRemove(refreshToken._id, { useFindAndModify: false }).exec();
    answer.set (AuthErrors.getError(AuthErrors.ERR_AUTH_REFRESH_EXPIRED));
    return next(answer);
  }

  // create the jwt token that it sent as a cookie. It contains a copy of the xsrf token
  let jwtToken = jwt.sign({ id: refreshToken.user._id.toString(), xsrfToken }, Config.secretKey, {
    expiresIn: Config.jwtExpiration
  });
  console.log("jwt token [refreshed] = "+jwtToken);

  res.cookie('access_token', jwtToken, {
    maxAge: Config.jwtExpiration*1000,
    httpOnly: true,
    secure: true,
    signed: true,
    sameSite: 'none'
  });

  // update refresh token
  let tokenStr = uuidv4();
  //console.log(tokenStr)
  try {
    await RefreshToken.findByIdAndUpdate(refreshToken._id, {token: tokenStr}).populate('user').exec();
  }
  catch(err) {
    console.log("cannot update refresh token")
  }

  answer.setPayload({
    refreshToken: tokenStr,
  });

  res.status(200).send(answer);
};

module.exports = {
  verifyToken,
  signIn,
  refreshToken,
}
