const User = require('../../models/auth/user.model')
const UserErrors = require('../../commons/user.errors')
const Hero = require('../../models/herocorp/heroe.model')

const Config = require('../../commons/config');

const axios = require('axios');
const bcrypt = require('bcryptjs');
const SALT_WORK_FACTOR = 10;

const {answer} = require('../ControllerAnswer')

/**
 * get an user with its login
 * @param {Object} req - The request object (provided by express)
 * @param {Object} res - The result object used to send the result to the client (provided by express)
 * @param {Function} next - The next middleware to call after this one
 */
const getUser = async function (req, res, next) {
  answer.reset()
  console.log('get user by login');
  let user = null
  try {
    user = await User.find({login: req.params.login}).populate('hero').exec();
  }
  catch(err) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_INVALID_FIND_USER_REQUEST))
    return next(answer);
  }
  // if none is found => error
  if ((user.length) && (user.length !== 1)) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_INVALID_FIND_USER_REQUEST))
    return next(answer);
  }
  // sends back all users
  answer.setPayload( user[0])
  res.status(200).send(answer)
};

const createUser = async function (req, res, next) {
  answer.reset()
  console.log('create an user');
  if ((!req.body.login) || (!req.body.password) || (!req.body.hero) || (!req.body.captchaToken)) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
    return next(answer);
  }

  // check the captcha
  let data = {
    secret: Config.captchaSecretKey,
    response: req.body.captchaToken
  };

  let ok = await axios.post('https://www.google.com/recaptcha/api/siteverify?secret='+data.secret+'&response='+data.response)
      .then( response => {
            //console.log(response);
            if (!response.data.success) {
              answer.set(UserErrors.getError(UserErrors.ERR_USER_CAPTCHATOKEN_FAILED))
            }
            return response.data.success;
          }
      )
      .catch( err => {
            //console.log(err);
            answer.set(UserErrors.getError(UserErrors.ERR_USER_CAPTCHATOKEN_FAILED))
            return false;
          }
      );

  if (!ok) {
    return next(answer);
  }


  // check if user's login already exists
  try {
    let user = await User.findOne({login:req.body.login}).exec();
    if (user !== null) {
      console.log("login already exists")
      answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
      return next(answer);
    }
  }
  catch(err) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
    return next(answer);
  }
  // check if hero public name exists in db
  let hero = null
  try {
    hero = await Hero.findOne({publicName: req.body.hero}).exec()
    if (hero === null) {
      console.log("cannot find the hero")
      answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
      return next(answer);
    }
  }
  catch(err) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
    return next(answer);
  }
  // check if that hero is not already bounded to another user
  try {
    let uu = await User.findOne({hero: hero._id}).exec()
    if (uu !== null) {
      console.log("hero already bounded to user")
      answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
      return next(answer);
    }
  }
  catch(err) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
    return next(answer);
  }
  // first encrypt password
  let password = '';
  try {
    const salt = await bcrypt.genSalt(SALT_WORK_FACTOR);
    password = await bcrypt.hash(req.body.password, salt);
  } catch (err) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
    return next(answer);
  }
  let u = {
    login: req.body.login,
    password: password,
    hero: hero._id.toString(),
  }
  const user = new User(u)
  try {
    user.save()
    console.log(user)
  }
  catch(err) {
    answer.set(UserErrors.getError(UserErrors.ERR_USER_CANNOT_CREATE_USER))
    return next(answer);
  }
  // sends back the new user
  answer.setPayload(user)
  res.status(200).send(answer)
};

module.exports = {
  getUser,
  createUser,
}