import axios from 'axios'
import { API_PATH } from '@/common/config'

// creation d'un agent axios, avec une config. pour atteindre l'API
export const axiosAgent = axios.create({
  baseURL: API_PATH,
})

/* handleError() permet de gérer tous les cas d'erreurs : code d'erreur de réponse http (4XX, 5XX), absence de réponse, erreru réseau, ...
   Cependant, afin de simplifier l'écriture du front, cette fonction transforme l'erreur axios en un objet tel que renvoyé par axios en cas de succès
   c.a.d. un JSON avec un champ data contenant la réponse. Or dans le cas présent, l'API renvoie toujours des objets au format {error: ..., data: ...}.
   Il faut donc intialiser ce champ data avec un objet ayant ce même format et le renovyer quel que soit le cas d'erreur. La façon d'initialiser
   cet objet dépend du type d'erreur (cf. ci-dessous)
 */

async function handleError(service, name, typeReq, err) {

  /* Cas n°1 : la requête a été reçue par le serveur mais celui-ci renvoie un status != 2XX, ce qui signifie
       une erreur. Par exemple, il peut renovyer un status 404 pour dire que la ressource demandée n'existe pas.
  */
  if (err.response) {
    console.log("ERROR calling service "+name+" -  ["+typeReq+"] " + service + ": " + JSON.stringify(err.response));
    // err.response contient des informations sur l'erreur. On peut donc s'en servir pour créer l'objet assigné au champ data.
    return {
      data: {
        error: err.response.data.error,
        data: err.response.data.data
      }
    }

  }
  /* Cas n°2 : la requete a été envoyée mais aucune réponse reçue.
   */
  else if (err.request) {
    console.log("NETWORK ERROR calling service "+name+" -  ["+typeReq+"] " + service + ": " + JSON.stringify(err.response));
    // cette fois, pas d'informations. On construit donc un objet type.
    return {
      data: {
        error: 2,
        data: 'Le serveur est injoignable ou l\'URL demandée n\'existe pas'
      }
    };
  }
  /* Autre cas : réseau injoignable, pb ssl, ...
   */
  else {
    console.log("UNKNOWN ERROR while calling SERVICE " + name);
    return {
      data: {
        error: 1,
        data: 'Erreur réseau'
      }
    };
  }
}

/* Fonctions génériques pour envoyer des requêtes http
NB1: ces fonctions n'échouent jamais et renvoient forcément un objet ayant la même structure que les données
renvoyées par l'API, même en cas d'erreur.
NB2: le paramètre name sert à donner un alias au service, en plus de service qui est la route d'accès. Cela permet
 de rendre les messages de déboguages plus explicites.
*/

async function getRequest(service, name) {
  console.log("GET route "+service)
  let response = null
  try {
    response = await axiosAgent.get(service)
  } catch (err) {
    // le catch se fait si le serveur répond avec une erreur type 4XX, 5XX, ou bien si le serveur est off
    // dans ce cas, on appelle la méthode pour traiter ces types d'erreurs
    response = await handleError(service, name, 'get', err);
  }
  return response.data;
}

async function postRequest(service, data, name, config = {}) {
  console.log("POST route "+service)
  let response = null
  try {
    response = await axiosAgent.post(service, data, config)
  } catch (err) {
    // le catch se fait si le serveur répond avec une erreur type 4XX, 5XX, ou bien si le serveur est off
    // dans ce cas, on appelle la méthode pour traiter ces types d'erreurs
    response = await handleError(service, name, 'post', err);
  }
  return response.data;
}

async function patchRequest(service, data, name, config = {}) {
  console.log("PATCH route "+service)
  let response = null
  try {
    response = await axiosAgent.patch(service, data, config)
  } catch (err) {
    // le catch se fait si le serveur répond avec une erreur type 4XX, 5XX, ou bien si le serveur est off
    // dans ce cas, on appelle la méthode pour traiter ces types d'erreurs
    response = await handleError(service, name, 'patch', err);
  }
  return response.data;
}

async function putRequest(service, data, name, config = {}) {
  console.log("PUT route "+service)
  let response = null
  try {
    response = await axiosAgent.put(service, data, config)
  } catch (err) {
    // le catch se fait si le serveur répond avec une erreur type 4XX, 5XX, ou bien si le serveur est off
    // dans ce cas, on appelle la méthode pour traiter ces types d'erreurs
    response = await handleError(service, name, 'put', err);
  }
  return response.data;
}

async function deleteRequest(service, data, name, config = {}) {
  console.log("DELETE route "+service)
  let response = null
  try {
    response = await axiosAgent.delete(service,  {...config, data: data})
  } catch (err) {
    // le catch se fait si le serveur répond avec une erreur type 4XX, 5XX, ou bien si le serveur est off
    // dans ce cas, on appelle la méthode pour traiter ces types d'erreurs
    response = await handleError(service, name, 'delete', err);
  }
  return response.data;
}

export {
  getRequest,
  postRequest,
  patchRequest,
  putRequest,
  deleteRequest
}

